
from fastapi import APIRouter, Depends, Body, UploadFile, Form, Request

from dbgpt.app.apps.utils.file.file_manage import FileManage
from dbgpt.app.apps.utils.response import SuccessResponse,ErrorResponse
from dbgpt.app.apps.vadmin.auth.utils.current import FullAdminAuth
from dbgpt.app.apps.vadmin.auth.utils.validation.auth import Auth
from . import schemas, crud
from .params.media_list import MediaListParams, GroupListParams, MediaEditParams, QuestionListParams, \
    QuestionEditParams, CorrelationListParams
from ...core.dependencies import IdList

router = APIRouter()



@router.post("/upload/image", summary="上传图片到本地")
async def upload_image_to_local(file: UploadFile, path: str = Form(...)):
    manage = FileManage(file, path)
    path = await manage.save_image_local()
    return SuccessResponse(path)

@router.post("/upload/video", summary="上传视频到本地")
async def upload_video_to_local(file: UploadFile, path: str = Form(...)):
    manage = FileManage(file, path)
    path = await manage.save_video_local()
    return SuccessResponse(path)

@router.post("/upload/save", summary="上传提交保存")
async def upload_save(data: schemas.Media, auth: Auth = Depends(FullAdminAuth())):
    return SuccessResponse(await crud.MediaDal(auth.db).create_data(data=data))

@router.get("/media/list", summary="资源列表(图片、视频)")
async def media_image_list(params: MediaListParams = Depends(), auth: Auth = Depends(FullAdminAuth())):
    datas, count = await crud.MediaDal(auth.db).get_datas(**params.dict(), v_return_count=True)
    return SuccessResponse(datas, count=count)

@router.post("/media/del", summary="删除资源")
async def media_edit(ids: IdList = Depends(), auth: Auth = Depends(FullAdminAuth())):
    await crud.MediaDal(auth.db).delete_datas(ids.ids, v_soft=True)
    return SuccessResponse("删除成功")


@router.post("/group/add", summary="新建分组")
async def group_add(data: schemas.Group, auth: Auth = Depends(FullAdminAuth())):
    return SuccessResponse(await crud.GroupDal(auth.db).create_data(data=data))

@router.get("/group/list", summary="分组列表")
async def image_list(params: GroupListParams = Depends(), auth: Auth = Depends(FullAdminAuth())):
    datas, count = await crud.GroupDal(auth.db).get_datas(**params.dict(), v_return_count=True)
    return SuccessResponse(datas, count=count)

@router.post("/group/del", summary="删除分组")
async def group_del(ids: IdList = Depends(), auth: Auth = Depends(FullAdminAuth())):
    # 校验分组是否为空
    media_counts = await crud.MediaDal(auth.db).get_count_by_groupIds(ids.ids)
    if media_counts > 0:
        return ErrorResponse("分组内不为空，无法删除")
    await crud.GroupDal(auth.db).delete_datas(ids.ids, v_soft=True)
    return SuccessResponse("删除成功")


@router.post("/media/edit", summary="编辑资源名称、归属分组")
async def media_edit(params: MediaEditParams = Depends(), auth: Auth = Depends(FullAdminAuth())):
    return SuccessResponse(await crud.MediaDal(auth.db).update_media_datas(params))


@router.post("/question/add", summary="新建问答对")
async def question_add(data: schemas.Question, auth: Auth = Depends(FullAdminAuth())):
    return SuccessResponse(await crud.QuestionDal(auth.db).create_data(data=data))


@router.get("/question/list", summary="问答对列表")
async def question_list(params: QuestionListParams = Depends(), auth: Auth = Depends(FullAdminAuth())):
    datas, count = await crud.QuestionDal(auth.db).get_datas(**params.dict(), v_return_count=True)
    return SuccessResponse(datas, count=count)


@router.post("/question/del", summary="删除问答对")
async def question_add(ids: IdList = Depends(), auth: Auth = Depends(FullAdminAuth())):
    return SuccessResponse(await crud.QuestionDal(auth.db).delete_datas(ids.ids, v_soft=True))


@router.post("/question/edit", summary="编辑问答对")
async def question_edit(params: QuestionEditParams = Depends(), auth: Auth = Depends(FullAdminAuth())):
    return SuccessResponse(await crud.QuestionDal(auth.db).update_question_datas(params))




@router.post("/correlation/edit", summary="编辑知识库关联关系")
async def correlation_add(data: schemas.Correlation, auth: Auth = Depends(FullAdminAuth())):
    return SuccessResponse(await crud.CorrelationDal(auth.db).put_data(data_id=data.id, data=data))


@router.get("/correlation/list", summary="资源关联列表")
async def correlation_list(params: CorrelationListParams = Depends(), auth: Auth = Depends(FullAdminAuth())):
    datas, count = await crud.CorrelationDal(auth.db).get_datas(**params.dict(), v_return_count=True)
    return SuccessResponse(datas, count=count)












