
from fastapi import APIRouter, Depends, Body, UploadFile, Form, Request

from dbgpt.app.apps.utils.file.file_manage import FileManage
from dbgpt.app.apps.utils.response import SuccessResponse,ErrorResponse
from dbgpt.app.apps.vadmin.auth.utils.current import FullAdminAuth
from dbgpt.app.apps.vadmin.auth.utils.validation.auth import Auth
from . import schemas, crud
from .models import VadminQuestion, VadminMedia
from .params.media_list import MediaListParams, GroupListParams, MediaEditParams, QuestionListParams, \
    QuestionEditParams, CorrelationListParams
from ...core.dependencies import IdList
from dbgpt.app.apps.utils.filter import mydfafiter, mydfafiter_picture, mydfafiter_question, mydfafiter_video
from sqlalchemy import BinaryExpression

router = APIRouter()



@router.post("/upload/image", summary="上传图片到本地")
async def upload_image_to_local(file: UploadFile, path: str = Form(...)):
    manage = FileManage(file, path)
    path = await manage.save_image_local()
    return SuccessResponse(path)

@router.post("/upload/video", summary="上传视频到本地")
async def upload_video_to_local(file: UploadFile, path: str = Form(...)):
    manage = FileManage(file, path)
    path = await manage.save_video_local()
    return SuccessResponse(path)

@router.post("/upload/save", summary="上传提交保存")
async def upload_save(data: schemas.Media, auth: Auth = Depends(FullAdminAuth())):

    if data.type == 1:
        # 新加一条图片的关键词
        mydfafiter_picture.add(data.key_word)
    elif data.type == 2:
        # 新加一条视频的关键词
        mydfafiter_video.add(data.key_word)

    return SuccessResponse(await crud.MediaDal(auth.db).create_data(data=data))

@router.get("/media/list", summary="资源列表(图片、视频)")
async def media_image_list(params: MediaListParams = Depends(), auth: Auth = Depends(FullAdminAuth())):
    datas, count = await crud.MediaDal(auth.db).get_datas(**params.dict(), v_return_count=True)
    return SuccessResponse(datas, count=count)

@router.post("/media/del", summary="删除资源")
async def media_edit(ids: IdList = Depends(), auth: Auth = Depends(FullAdminAuth())):

    id_filter: BinaryExpression = VadminMedia.id.in_(ids.ids)
    media_datas, count = await crud.MediaDal(auth.db).get_datas(v_where=[id_filter], v_return_count=True)
    for qdata in media_datas:
        if qdata.get('type') == 1:
            mydfafiter_picture.remove(qdata.get('key_word'))
        elif qdata.get('type') == 2:
            mydfafiter_video.remove(qdata.get('key_word'))

    await crud.MediaDal(auth.db).delete_datas(ids.ids, v_soft=True)
    return SuccessResponse("删除成功")


@router.post("/group/add", summary="新建分组")
async def group_add(data: schemas.Group, auth: Auth = Depends(FullAdminAuth())):
    return SuccessResponse(await crud.GroupDal(auth.db).create_data(data=data))

@router.get("/group/list", summary="分组列表")
async def image_list(params: GroupListParams = Depends(), auth: Auth = Depends(FullAdminAuth())):
    datas, count = await crud.GroupDal(auth.db).get_datas(**params.dict(), v_return_count=True)
    return SuccessResponse(datas, count=count)

@router.post("/group/del", summary="删除分组")
async def group_del(ids: IdList = Depends(), auth: Auth = Depends(FullAdminAuth())):
    # 校验分组是否为空
    media_counts = await crud.MediaDal(auth.db).get_count_by_groupIds(ids.ids)
    if media_counts > 0:
        return ErrorResponse("分组内不为空，无法删除")
    await crud.GroupDal(auth.db).delete_datas(ids.ids, v_soft=True)
    return SuccessResponse("删除成功")


@router.post("/media/edit", summary="编辑资源名称、归属分组")
async def media_edit(params: MediaEditParams = Depends(), auth: Auth = Depends(FullAdminAuth())):
    media_dic = {'id': params.ids[0]}
    media_datas = await crud.MediaDal(auth.db).get_datas(**media_dic)
    if len(media_datas) > 0:
        print(f"------编辑时要删除旧的media资源------>:{media_datas[0].get('key_word')}")
        print(f"------编辑时要添加新的media资源------>:{params.key_word}")
        if media_datas[0].get('type') == 1:
            mydfafiter_picture.remove(media_datas[0].get('key_word'))
            mydfafiter_picture.add(params.key_word)
        elif media_datas[0].get('type') == 2:
            mydfafiter_video.remove(media_datas[0].get('key_word'))
            mydfafiter_video.add(params.key_word)

    return SuccessResponse(await crud.MediaDal(auth.db).update_media_datas(params))


@router.post("/question/add", summary="新建问答对")
async def question_add(data: schemas.Question, auth: Auth = Depends(FullAdminAuth())):
    ret = await crud.QuestionDal(auth.db).create_data(data=data)
    if len(ret) > 0:
        print(f"-------------------------add->:{ret}")
        mydfafiter_question.add(data.key_word)  # 向关键词字典中添加一个新的问答对
    return SuccessResponse(ret)


@router.get("/question/list", summary="问答对列表")
async def question_list(params: QuestionListParams = Depends(), auth: Auth = Depends(FullAdminAuth())):
    datas, count = await crud.QuestionDal(auth.db).get_datas(**params.dict(), v_return_count=True)
    return SuccessResponse(datas, count=count)


@router.post("/question/del", summary="删除问答对")
async def question_add(ids: IdList = Depends(), auth: Auth = Depends(FullAdminAuth())):
    ret = await crud.QuestionDal(auth.db).delete_datas(ids.ids, v_soft=True)

    id_filter: BinaryExpression = VadminQuestion.id.in_(ids.ids)
    question_datas, count = await crud.QuestionDal(auth.db).get_datas(v_where=[id_filter], v_return_count=True)
    for qdata in question_datas:
        mydfafiter_question.remove(qdata.get('key_word'))

    return SuccessResponse(ret)


@router.post("/question/edit", summary="编辑问答对")
async def question_edit(params: QuestionEditParams = Depends(), auth: Auth = Depends(FullAdminAuth())):
    if len(params.key_word) > 0:
        question_dic = {'page': 1, 'limit': 0, 'v_order': None, 'v_order_field': None, 'id': params.ids[0]}
        question_datas, count = await crud.QuestionDal(auth.db).get_datas(**question_dic, v_return_count=True)
        if len(question_datas) > 0:
            print(f"------编辑时要删除旧的问答对------>:{question_datas[0]}")
            mydfafiter_question.remove(question_datas[0].get('key_word'))
            print(f"------编辑时要添加新的问答对------>:{params.key_word}")
            mydfafiter_question.add(params.key_word)
    ret = await crud.QuestionDal(auth.db).update_question_datas(params)
    return SuccessResponse(ret)




@router.post("/correlation/edit", summary="编辑知识库关联关系")
async def correlation_add(data: schemas.Correlation, auth: Auth = Depends(FullAdminAuth())):
    return SuccessResponse(await crud.CorrelationDal(auth.db).put_data(data_id=data.id, data=data))


@router.get("/correlation/list", summary="资源关联列表")
async def correlation_list(params: CorrelationListParams = Depends(), auth: Auth = Depends(FullAdminAuth())):
    datas, count = await crud.CorrelationDal(auth.db).get_datas(**params.dict(), v_return_count=True)
    return SuccessResponse(datas, count=count)












